#!/bin/ksh

export LANG=C
PY_VER=3.12

# This scripts is used to install the exracted dnf rpm packages.

if [[ -e /usr/opt/rpm/bin/rpm ]]
then
    RPM_CMD="/usr/opt/rpm/bin/rpm"
else
    RPM_CMD="/usr/bin/rpm"
fi

yum4=0
yum3_instd=0

if [[  "$#" -eq 4 ]] && [[ $4 -eq 2 ]] # Called from dnf_aixtoolbox.sh
then
    yum4=$2
    yum3_instd=$3
else # Called directly
    # Check if we are running this as the root user.
    if [[ "$(id -u)" != "0" ]]
    then
        echo "This script must be run as root."
        exit 1
    fi

    oslvl=`/usr/bin/oslevel`
    if [[ "$oslvl" = "6.1.0.0" ]]
    then
        echo "dnf and dependencies can be installed on AIX 7.1 and higher versions."
        exit 1
    fi

    # Check if yum3 is installed.
    yum3_instd=0
    $RPM_CMD -qa | /usr/bin/grep yum-3.4.3 > /dev/null 2>&1
    if [[ $? -eq 0 ]]
    then
        yum3_instd=1
    fi

    prog=${0##*/}
    yum4=0

    usage() {
        print >&2 "Usage: $prog <-d> <-y> <-n> -?

          -d    Install and setup dnf if yum is not installed.
          -y    Update yum3 to yum4(softlink to DNF).
            Use this option if want to redirect all yum commands to dnf.
            Existing yum-3.4.3 will be updated to yum-4.x.x(dnf)
          -n    Install dnf where both yum and dnf can coexist if yum is installed already.
                This is not a recommended option."
        exit 1
    }

    if [[ $# -ne 1 ]]
    then
        usage
        exit 1
    fi 
    arg=`echo $1 | /usr/bin/cut -c1-3`
    if [[ "$arg" = "-d" ]]
    then
        yum4=1 # Install only dnf if no YUM is installed.
        if [[ $yum3_instd -eq 1 ]]
        then
            echo "YUM is already installed in the machine."
            echo "Please use the option -y to update to YUM4(dnf)."
            exit 1
        fi
    elif [[ "$arg" = "-y" ]]
    then
        yum4=2 # Update existing YUM3 to YUM4.
    elif [[ "$arg" = "-n" ]]
    then
        yum4=3 # Have both YUM and dnf at the same time.
    else
        usage
    fi

    # Check openssl version.
    function print_openssl_err {
        echo "Please install openssl 1.1.x and higher version."
        echo "You can download and install latest openssl from AIX web download site"
        echo "https://www-01.ibm.com/marketing/iwm/platform/mrs/assets?source=aixbp"
        exit 1
    }
    ssl_ver=$(lslpp -Lc openssl.base | /usr/bin/awk 'FNR==2' | /usr/bin/awk -F':' '{print $3}')
    f1=$(echo $ssl_ver | /usr/bin/cut -d"." -f1)
    f2=$(echo $ssl_ver | /usr/bin/cut -d"." -f2)
    f3=$(echo $ssl_ver | /usr/bin/cut -d"." -f3)
    f4=$(echo $ssl_ver | /usr/bin/cut -d"." -f4)
    if [[ ( ! $f1 -ge 3 ) ]] || [[ ( $f1 -eq  0 ) &&  ( ! $f2 -ge 15 ) ]]
    then
        print_openssl_err
    fi

    # Check if /opt is having enough space to install the packages from dnf_bundle.
    # Currently we need around 457MB of free space in /opt filesystem.
    typeset -i total_opt=`echo "(1024)" | bc`
    opt_free=`df -m /opt | /usr/bin/sed -e /Filesystem/d | /usr/bin/head -1 | /usr/bin/awk '{print $3}'`
    if [[ $opt_free -le $total_opt ]]
    then
        echo "Total free space required for /opt filesystem to install rpms"
        echo "  from dnf_bundle is around 1024MB."
        echo "Please increase the size of /opt and retry."
        exit 1
    fi
fi


# Update rpm.rte to latest version.
# From AIX 7.1 TL5 & 7.2 TL2 rpm.rte shipped is 4.13.0.1.
# Installation will be skipped if either 4.9.1.3 or 4.13.0.1 is installed. 
echo ""
echo "Installing rpm.rte at the latest version ..."
echo "This may take several minutes depending on the number of rpms installed..."
installp -qacXYd rpm.rte all
if [[ $? -ne 0 ]]
then
    echo "rpm.rte update to latest version failed."
    cd - >/dev/null 2>&1
    #rm -rf $tmppath
    exit 1
fi

#rpm signature specific changes
$RPM_CMD --import RPM-GPG-KEY-IBM-AIX-Toolbox
if [[ $? -ne 0 ]]
then
    echo ""
    echo "importing AIX toolbox public gpg key is failed."
fi

echo "Verifying rpm packages signature..."
echo ""
$RPM_CMD -K *.rpm
if [[ $? -ne 0 ]]
then
    echo "rpm packages signature verification failed."
    echo ""
else
    echo "\n"
    echo "All rpm packages signature verified successfully."
fi


# Check if some packages are already installed from the dnf_bundle.
echo "\n"
echo "Checking whether any of the rpms from dnf_bundle are already installed..."
echo "Takes couple of minutes to process.\n"

if [[ ! -e extra_list ]]
then
    echo "Missing extra_list file from the bundle"
    exit 1
fi

rpmlib32_rpm4151_rpm=""
rpmlib32_rpm4151_rpm=`ls rpmlibs32-4.15.1.0*.rpm`

#Check if cacerts.base is installed. Don't install ca-certificates if cacerts.base is installed.
cacert_inst=$(ODMDIR=/usr/lib/objrepos /usr/bin/odmget -q "name='cacerts.base'" lpp)

extra_pkgs=$(cat extra_list)
set -A rpms_list
set -A pkg_list
set -A pkgname_list
set -A pkg_name_ver_list
set -A install_list

if [[ $yum4 -eq 1 ]] # Install only dnf if no YUM is installed.
then
    /usr/bin/ls *.rpm | /usr/bin/egrep -v "yum|python3-dnf-plugin-migrate|python3-dnf-plugins-core|\
python${PY_VER}-dnf-plugin-migrate|python${PY_VER}-dnf-plugins-core|\
python3-dateutil|python${PY_VER}-dateutil|python3-six|python${PY_VER}-six|\
$rpmlib32_rpm4151_rpm" | while read rpm_file
    do
        if [[ -n "$cacert_inst" ]]
        then
             echo $rpm_file | /usr/bin/grep "ca-certificates" > /dev/null 2>&1
             if [[ $? -eq 0 ]]
             then
                  continue
             fi
        fi 
        pkg_list[${#pkg_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}-%{VERSION}-%{RELEASE}.%{ARCH}" $rpm_file`
        pkgname_list[${#pkgname_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}" $rpm_file`
        rpms_list[${#rpms_list[*]}]=`echo $rpm_file`
        pkg_name_ver_list[${#pkg_name_ver_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}-%{VERSION}-%{RELEASE}" $rpm_file`
    done
elif [[ $yum4 -eq 2 ]] || [[ $yum4 -eq 3 ]] # Install yum4 which is a softlink to dnf.
then
   if [[ $yum3_instd -eq 0 ]] #If yum3 isn't installed then no need to install migrate related packages.
   then
       /usr/bin/ls *.rpm | /usr/bin/egrep -v "python3-dnf-plugin-migrate|python3-dnf-plugins-core|\
python${PY_VER}-dnf-plugin-migrate|python${PY_VER}-dnf-plugins-core|\
python3-dateutil|python${PY_VER}-dateutil|python3-six|python${PY_VER}-six|\
$rpmlib32_rpm4151_rpm" | while read rpm_file
       do
            if [[ -n "$cacert_inst" ]]
            then
                 echo $rpm_file | /usr/bin/grep "ca-certificates" > /dev/null 2>&1
                 if [[ $? -eq 0 ]]
                 then
                      continue
                 fi
            fi 
			# Check if dnf which obsoletes YUM is already installed.
			$RPM_CMD -q --obsoletes dnf | /usr/bin/grep yum > /dev/null 2>&1
			if [[ $? -eq 0 ]]
			then
				if [[ "$rpm_file" == *+(yum-)* ]]
				then
					continue
				fi
			fi	
            pkg_list[${#pkg_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}-%{VERSION}-%{RELEASE}.%{ARCH}" $rpm_file`
            pkgname_list[${#pkgname_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}" $rpm_file`
            rpms_list[${#rpms_list[*]}]=`echo $rpm_file`
            pkg_name_ver_list[${#pkg_name_ver_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}-%{VERSION}-%{RELEASE}" $rpm_file`
      done 
    else
        /usr/bin/ls *.rpm | /usr/bin/egrep -v "yum" | while read rpm_file
        do
            if [[ -n "$cacert_inst" ]]
            then
                 echo $rpm_file | /usr/bin/grep "ca-certificates" > /dev/null 2>&1
                 if [[ $? -eq 0 ]]
                 then
                      continue
                 fi
            fi 
            pkg_list[${#pkg_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}-%{VERSION}-%{RELEASE}.%{ARCH}" $rpm_file`
            pkgname_list[${#pkgname_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}" $rpm_file`
	        rpms_list[${#rpms_list[*]}]=`echo $rpm_file`
	        pkg_name_ver_list[${#pkg_name_ver_list[*]}]=`$RPM_CMD -qp --qf "%{NAME}-%{VERSION}-%{RELEASE}" $rpm_file`
        done
    fi
fi

# Compares the two packages version number
function cmp_version {
   large=$(echo  $1 $2 |
         /usr/bin/awk '{ split($1, a, ".");
         split($2, b, ".");
         x = 0;
         for (i = 1; i <= 4; i++)
            if (a[i] < b[i]) {
                x = 3;
                break;
            } else if (a[i] > b[i]) {
                x = 2;
                break;
            }
           print x;
         }')
   return $large
}

# Compares the two packages release number
function cmp_release {
   arg1=`expr $1`
   arg2=`expr $2`
   echo $1 | /usr/bin/grep "_" > /dev/null 2>&1
   ret1=$?
   echo $2 | /usr/bin/grep "_" > /dev/null 2>&1
   ret2=$?
   if [[ $ret1 -eq 0 ]] && [[ $ret2 -eq 0 ]]
   then
       af1=`echo $arg1 | /usr/bin/cut -d'_' -f1`
       af2=`echo $arg1 | /usr/bin/cut -d'_' -f2`
       bf1=`echo $arg2 | /usr/bin/cut -d'_' -f1`
       bf2=`echo $arg2 | /usr/bin/cut -d'_' -f2`
       if [[ $af1 -lt  $bf1 ]]
       then
          return 3
       elif [[ $af1 -gt  $bf1 ]]
       then
          return 2
       elif [[ $af1 -eq $bf1 ]]
       then
           if [[ $af2 -lt $bf2 ]]
           then
                return 3
            elif [[ $af2 -gt  $bf2 ]]
            then
                return 2
            fi
        fi
   else
       echo $2 | /usr/bin/grep "_" > /dev/null 2>&1
       if [[ $? -eq 0 ]]
       then
           arg2=`echo $2  | /usr/bin/grep "_" | /usr/bin/sed 's/_.*//' | bc`
       fi
       if [[ $arg1 -lt $arg2 ]]
       then
          return 3
       elif [[ $arg1 -gt  $arg2 ]]
       then
          return 2
       fi
    fi
}

let "pkg_count=0"
pkg_count=${#pkg_list[@]}
p=0
while (( $p < $pkg_count ))
do
    $RPM_CMD -q --qf %{NAME} ${pkgname_list[$p]} > /dev/null 2>&1
    n_inst=$?
    if [[ $n_inst -eq 0 ]] # package with the same name is installed
    then
	    inst_pkgname=`$RPM_CMD -q --qf %{NAME} ${pkgname_list[$p]}` 
	    $RPM_CMD -q --qf %{NAME}-%{VERSION}-%{RELEASE}.%{ARCH} ${pkg_list[$p]} > /dev/null 2>&1
	    i_inst=$?
	    if [[ $i_inst -ne  0 ]] # package with exact version isn't installed
	    then
	        # check if rpm from dnf_bundle is higher version than the one installed.
	        inst_pkgname_ver=`$RPM_CMD -q --qf %{NAME}-%{VERSION}-%{RELEASE}.%{ARCH} ${pkgname_list[$p]}`
	        if [[ "${pkgname_list[$p]}" = "$inst_pkgname" ]]
	        then
		        inst_ver=`$RPM_CMD -q --qf %{VERSION} ${pkgname_list[$p]}`
		        rpm_ver=`$RPM_CMD -qp --qf %{VERSION} ${rpms_list[$p]}`
		        cmp_version $inst_ver $rpm_ver
		        rc1=$?
		        if [[ $rc1 -eq 0 ]] # Same version matches. Check for the release.
		        then
		            inst_rel=`$RPM_CMD -q --qf %{RELEASE} ${pkgname_list[$p]}`
		            rpm_rel=`$RPM_CMD -qp --qf %{RELEASE} ${rpms_list[$p]}`
		            cmp_release $inst_rel $rpm_rel 
		            rc2=$?
		            if [[ $rc2 -eq 3 ]] # Installed release is less than the bundle 
		            then
			            echo "$inst_pkgname_ver will be updated to ${pkg_name_ver_list[$p]}" 
			            install_list[${#install_list[*]}+1]=${rpms_list[$p]}
		            elif [[ $rc2 -eq 2 ]] # Higher version is already installed
		            then
		   	            echo "Skipping ${pkg_name_ver_list[$p]} as higher version $inst_pkgname_ver is installed."
		            fi
		        elif [[ $rc1 -eq 3 ]] # Installed rpm lower than the bundle
		        then
		            echo "$inst_pkgname_ver will be updated to ${pkg_name_ver_list[$p]}"
		            install_list[${#install_list[*]}+1]=${rpms_list[$p]}
		        elif [[ $rc1 -eq 2 ]]
		        then
		            echo "Skipping ${pkg_name_ver_list[$p]} as higher version $inst_pkgname_ver is installed."
		        fi
	        fi
	    elif [[ $i_inst -eq 0 ]] # exact version of the package is installed. Do nothing.
	    then
	        echo "${pkg_name_ver_list[$p]} is already installed" 
	    fi
    elif [[ $n_inst -ne 0 ]] # package with the name isn't installed
    then
        # Install only required packages, skip sub-packages install
        # if python3 meta package is installed then need to take care of installing versioned packages as well.
        echo "${pkg_name_ver_list[$p]}" | /usr/bin/egrep "python${PY_VER}-devel-${PY_VER}\.*|\
python${PY_VER}-idle-${PY_VER}\.*|python${PY_VER}-test-${PY_VER}\.*|python${PY_VER}-tkinter-${PY_VER}\.*" > /dev/null 2>&1
        if [[ $? -eq 0 ]]
        then
            py3_name=`$RPM_CMD -qp --qf %{NAME} ${rpms_list[$p]}|/usr/bin/sed 's/3\.9/3/g'`
            $RPM_CMD -q $py3_name > /dev/null 2>&1
            if [[ $? -eq 0 ]]
            then
                echo "${pkg_name_ver_list[$p]} will be installed"
                install_list[${#install_list[*]}+1]=${rpms_list[$p]}
            fi
		fi
        echo $extra_pkgs | /usr/bin/grep "${pkg_name_ver_list[$p]}" > /dev/null 2>&1
        if [[ $? -ne 0 ]]
        then
             echo "${pkg_name_ver_list[$p]} will be installed"
             install_list[${#install_list[*]}+1]=${rpms_list[$p]}
         fi
    fi
    (( p+=1 ))
done 

if [[ ${#install_list[@]} -eq 0 ]]
then
    echo "\ndnf and all it's dependencies are already installed."
    cd - >/dev/null 2>&1
    #rm -rf $tmppath
    exit 0
fi

echo "\nInstalling the packages...\n"
rpm -Uvh ${install_list[@]}
rc=$?

if [[ $rc -eq 0 ]]
then
    cd - >/dev/null 2>&1
    #rm -rf $tmppath
    echo
    echo "\033[1;32mdnf installed successfully. \033[m"
    echo "\033[1;33mPlease run 'dnf update' to update packages to the latest level. \033[m"
    echo
    echo "Please note, RPM packages are downloaded in dnf cache /var/cache/dnf."
    echo "RPM packages install files go under the path /opt."
    echo "Hence it is recommended to always keep at least 512MB of free space in /var & /opt" 
    echo "to avoid any download and installation/update failures."
    echo
    if [[ ( $yum4 -eq 2 ) || ( $yum4 -eq 3 ) ]] && [[ $yum3_instd -eq 1 ]] # Run this when yum3 is installed.
    then
        /usr/bin/echo ""
        /usr/bin/echo "Trying to create pkg_yumdb database required for migrate plugin"
        LIBPATH=/opt/freeware/lib/librpm32:/opt/freeware/lib:/usr/opt/rpm/lib:/usr/lib:/lib /opt/freeware/bin/python2\
                 /opt/freeware/sbin/update_pkg_yumdb
        if [[ $? -ne 0 ]]
        then
            /usr/bin/echo "Failed to create pkg_yumdb database"
        else
            /usr/bin/echo "Successfully created pkg_yumdb database"
            echo
            echo "Trying to run dnf migrate to migrate yum database to dnf."
            # Make sure /var/lib/dnf/history.sqlite isn't present.
            if [[ -e /var/lib/dnf/history.sqlite ]]
            then
                /usr/bin/mv /var/lib/dnf/history.sqlite /var/lib/dnf/history.sqlite_bak_mig
            fi
            /opt/freeware/bin/dnf migrate
            # Remove the plugins related packages
            if [[ $? -eq 0 ]]
            then
               echo
               echo "Successfully migrated yum database to dnf."
               echo
               echo "Trying to remove plugins related packages"
               rpm -evh  python3-dnf-plugin-migrate python3-dnf-plugins-core\
                       python${PY_VER}-dnf-plugin-migrate python${PY_VER}-dnf-plugins-core python3-dateutil\
                       python${PY_VER}-dateutil python3-six python${PY_VER}-six rpmlibs32
            fi
        fi
        # Now try to install yum-4
        rpm -Uvh yum-*.rpm
    fi
elif [[ $rc -ne 0 ]]
then
    echo
    echo "\033[1;31mdnf installation failed. \033[m" 
    echo "If the failure was due to a space issue, increase the size of /opt and re-run script again."
    echo "Another reason for failure could be mixing of Toolbox packages and packages from other sources."
    echo
    $RPM_CMD -q python3-docs > /dev/null 2>&1
    if [[ $? -eq 0 ]]
    then
        echo "Please remove python3-docs manually and retry the dnf install."
        echo "python${PY_VER} doesn't have python3-docs as a sub-packages."
        echo
    fi
    exit 1
fi
exit 0
